%feature("docstring") OT::Ipopt
R"RAW(Ipopt nonlinear optimization solver.

`Ipopt <https://coin-or.github.io/Ipopt/>`_ is a software package for large-scale nonlinear optimization.

Parameters
----------
problem : :class:`~openturns.OptimizationProblem`, optional
    Optimization problem to solve. Default is an empty problem.

Notes
-----
*Algorithms parameters:*

  Ipopt algorithms can be adapted using numerous parameters, described  `here <https://coin-or.github.io/Ipopt/OPTIONS.html>`_.
  These parameters can be modified using the :class:`~openturns.ResourceMap`.
  For every option ``optionName``, one simply add a key named ``Ipopt-optionName`` with the value to use, as shown below::

    >>> import openturns as ot
    >>> ot.ResourceMap.AddAsUnsignedInteger('Ipopt-print_level', 5)
    >>> ot.ResourceMap.AddAsScalar('Ipopt-diverging_iterates_tol', 1e15)

*Convergence criteria:*

  To estimate the convergence of the algorithm during the optimization process, Ipopt uses specific tolerance parameters, different from the standard absolute/relative/residual errors used in OpenTURNS. The definition of Ipopt's parameters can be found in `this paper, page 3 <http://www.optimization-online.org/DB_FILE/2004/03/836.pdf>`_.

  Thus the attributes ``maximumAbsoluteError``, ``maximumRelativeError``, ``maximumResidualError`` and ``maximumConstraintError`` defined in' ``OptimizationAlgorithm`` are not used in this case. The tolerances used by Ipopt can be set using specific options (e.g. ``tol``, ``dual_inf_tol`` ...).


See also
--------
Bonmin

Examples
--------
The code below ensures the optimization of the following problem:

.. math:: min \left( - (x_2-2)^2 \right)

subject to

.. math::

    x_1 \times x_1 + x_2 = 1

    x_1 \in (-1, 1)

    x_2 \in \mathbb{R}


>>> import openturns as ot
>>> objective = ot.SymbolicFunction(['x1','x2'], ['-(x2 - 2.0)^2'])
>>> bounds = ot.Interval([-1, -1e19], [1, 1e19])
>>> g = ot.SymbolicFunction(['x1','x2'], ['-(x1 * x1 + x2 - 1.0)'])
>>> problem = ot.OptimizationProblem(objective)
>>> problem.setBounds(bounds)
>>> problem.setEqualityConstraint(g)

>>> algo = ot.Ipopt(problem)
>>> algo.setStartingPoint([0.5, 1.5])
>>> algo.run() # doctest: +SKIP

>>> # Retrieving the results
>>> result = algo.getResult() # doctest: +SKIP
>>> optimalPoint = result.getOptimalPoint() # doctest: +SKIP
>>> optimalValue = result.getOptimalValue() # doctest: +SKIP
>>> evaluationNumber = result.getInputSample().getSize() # doctest: +SKIP )RAW"

