%feature("docstring") OT::GeneralLinearModelAlgorithm
R"RAW(Algorithm for the evaluation of general linear models.

Refer to :ref:`gaussian_process_regression`.

Available constructors:
    GeneralLinearModelAlgorithm(*inputSample, outputSample, covarianceModel, keepCovariance=True*)

    GeneralLinearModelAlgorithm(*inputSample, outputSample, covarianceModel, basis, keepCovariance=True*)

Parameters
----------
inputSample, outputSample : 2-d sequence of float
    The samples :math:`(\vect{x}_k)_{1 \leq k \leq \sampleSize} \in \Rset^\inputDim` and
    :math:`(\vect{y}_k)_{1 \leq k \leq \sampleSize}\in \Rset^\outputDim`.

covarianceModel : :class:`~openturns.CovarianceModel`
    Covariance model of the Gaussian process. See notes for the details.

basis : :class:`~openturns.Basis`
    Functional basis to estimate the trend: :math:`(\varphi_j)_{1 \leq j \leq n_1}: \Rset^\inputDim \rightarrow \Rset`.
    If :math:`\outputDim > 1`, the same basis is used for each marginal output.

keepCovariance : bool, optional
    Indicates whether the covariance matrix has to be stored in the result structure *GeneralLinearModelResult*.
    Default value is set in resource map key `GeneralLinearModelAlgorithm-KeepCovariance`

Notes
-----
We suppose we have a sample :math:`(\vect{x}_k, \vect{y}_k)_{1 \leq k \leq \sampleSize}` where
:math:`\sampleSize \in \Nset` is the sample size, :math:`\vect{y}_k = \model(\vect{x}_k)` for all :math:`k \in \{1, ..., \sampleSize\}`
and :math:`\model:\Rset^\inputDim \mapsto \Rset^\outputDim` is the model.
The objective is to build a metamodel :math:`\metaModel`, using a **general linear model**: the sample
:math:`(\vect{y}_k)_{1 \leq k \leq \sampleSize}` is considered as the restriction of a Gaussian process
:math:`\vect{Y}(\omega, \vect{x})` on :math:`(\vect{x}_k)_{1 \leq k \leq \sampleSize}`.
The Gaussian process :math:`\vect{Y}(\omega, \vect{x})` is defined by:

.. math::

    \vect{Y}(\omega, \vect{x}) = \vect{\mu}(\vect{x}) + \vect{W}(\omega, \vect{x})

where:

.. math::

    \vect{\mu}(\vect{x}) = \left(
      \begin{array}{l}
        \mu_1(\vect{x}) \\
        \vdots  \\
        \mu_\outputDim(\vect{x})
       \end{array}
     \right)

with :math:`\mu_\ell(\vect{x}) = \sum_{j=1}^{n_\ell} \beta_j^\ell \varphi_j^\ell(\vect{x})`
for any :math:`\ell \in \{1, ..., \outputDim\}`,
where :math:`n_\ell \in \Nset` is the number trend functions of the :math:`\ell`-th output component, and
:math:`\varphi_j^\ell: \Rset^\inputDim \rightarrow \Rset` the trend functions.
The Gaussian process :math:`\vect{W}` is of dimension :math:`\outputDim` with zero mean and covariance function
:math:`C = C(\vect{\theta}, \vect{\sigma}, \mat{R}, \vect{\lambda})` (see :ref:`process_definitions`
for the notations).

We note:

.. math::

    \vect{\beta}^\ell = \left(
      \begin{array}{l}
        \beta_1^\ell \\
        \vdots  \\
        \beta_{n_\ell}^\ell
       \end{array}
     \right) \in \Rset^{n_\ell}
     \quad \mbox{ and } \quad
     \vect{\beta} = \left(
      \begin{array}{l}
         \vect{\beta}^1\\
         \vdots  \\
         \vect{\beta}^\outputDim
       \end{array}
     \right)\in \Rset^{n_t}

where :math:`n_t = \sum_{\ell = 1}^p n_\ell` is the total number of trend functions for all
output dimensions.

The *GeneralLinearModelAlgorithm* class estimates the coefficients
:math:`(\beta_j^\ell)_{j=1, ..., n_\ell, \; \ell=1, ..., \outputDim}` and :math:`\vect{p}`
where :math:`\vect{p}` is the vector of parameters of the covariance model, i.e. a subset of
:math:`\vect{\theta}, \vect{\sigma}, \mat{R}, \vect{\lambda}`, that has been declared as
*active* (by default, the full vectors :math:`\vect{\theta}` and :math:`\vect{\sigma}`).
The estimation is done by maximizing the *reduced* log-likelihood of the model
(see the equation :eq:`logLikelihood` below).

**Estimation of the parameters** :math:`\beta_j^\ell` and :math:`\vect{p}`

We note:

.. math::

    \vect{y} = \left(
      \begin{array}{l}
        \vect{y}_1 \\
        \vdots  \\
        \vect{y}_\sampleSize
       \end{array}
     \right) \in \Rset^{\sampleSize \outputDim},
     \quad
     \vect{m}_{\vect{\beta}} = \left(
      \begin{array}{l}
        \vect{\mu}(\vect{x}_1) \\
        \vdots  \\
        \vect{\mu}(\vect{x}_\sampleSize)
       \end{array}
     \right) \in \Rset^{\sampleSize \outputDim}

and

.. math::

    \mat{C}_{\vect{p}} = \left(
      \begin{array}{lcl}
        \mat{C}_{11} & \dots &  \mat{C}_{1 \sampleSize}\\
        \vdots       &       & \vdots \\
        \mat{C}_{\sampleSize1} & \dots &  \mat{C}_{\sampleSize\sampleSize}
       \end{array}
     \right) \in \cS_{\sampleSize \outputDim}^+(\Rset)

where :math:`\mat{C}_{ij} = C_{\vect{p}}(\vect{x}_i, \vect{x}_j)`.
The model likelihood writes:

.. math::

    \cL(\vect{\beta}, \vect{p};(\vect{x}_k, \vect{y}_k)_{1 \leq k \leq \sampleSize}) 
    = \dfrac{1}{(2\pi)^{\frac{1}{2} \sampleSize \outputDim} \left|\det\left(\mat{C}_{\vect{p}}\right)\right|^{\frac{1}{2}}}
    \exp\left( -\dfrac{1}{2}\Tr{\left( \vect{y}-\vect{m} \right)} \mat{C}_{\vect{p}}^{-1}  \left( \vect{y}-\vect{m} \right)  \right)

Let :math:`\mat{L}_{\vect{p}}` be the Cholesky factor of :math:`\mat{C}_{\vect{p}}`, i.e. the lower triangular matrix
with positive diagonal such that :math:`\mat{L}_{\vect{p}} \,\Tr{\mat{L}_{\vect{p}}} = \mat{C}_{\vect{p}}`.
Therefore the log-likelihood is:

.. math::
    :label: logLikelihood

    \log \cL(\vect{\beta}, \vect{p};(\vect{x}_k, \vect{y}_k)_{1 \leq k \leq \sampleSize}) 
    = \alpha - \log \left( \det \left(\mat{L}_{\vect{p}}\right) \right)
    -\dfrac{1}{2}  \left\| \mat{L}_{\vect{p}}^{-1}(\vect{y}-\vect{m}_{\vect{\beta}}) \right\|^2_2

where :math:`\alpha \in \Rset` is a constant independent of :math:`\vect{\beta}` and :math:`\vect{p}`.
The maximization of :eq:`logLikelihood` leads to the following optimality condition for :math:`\vect{\beta}`:

.. math::

    \vect{\beta}^*(\vect{p}^*)
    = \argmin_{\vect{\beta}} \left\| \mat{L}_{\vect{p}^*}^{-1} \left(\vect{y} - \vect{m}_{\vect{\beta}} \right) \right\|^2_2

This expression of :math:`\vect{\beta}^*` as a function of :math:`\vect{p}^*` is taken as a
general relation between :math:`\vect{\beta}` and :math:`\vect{p}` and is substituted into
:eq:`logLikelihood`, leading to a *reduced log-likelihood* function depending solely on :math:`\vect{p}`.

In the particular case where :math:`\outputDim = \dim(\vect{\sigma}) = 1` and :math:`\sigma` is a part of
:math:`\vect{p}`, then a further reduction is possible. In this case, let :math:`\vect{q}` be
the vector :math:`\vect{p}` in which :math:`\sigma` has been substituted by 1.
Therefore:

.. math::

    \left\| \mat{L}_{\vect{p}}^{-1}(\vect{y}-\vect{m}_{\vect{\beta}}) \right\|^2 
    = \frac{1}{\sigma^2} \left\| \mat{L}_{\vect{q}}^{-1} \left(\vect{y} - \vect{m}_{\vect{\beta}} \right) \right\|^2_2.

This shows that :math:`\vect{\beta}^*` is a function of :math:`\vect{q}^*` only, and the optimality
condition for :math:`\sigma` reads:

.. math::

    \vect{\sigma}^*(\vect{q}^*)
    =\dfrac{1}{\sampleSize} 
    \left\| \mat{L}_{\vect{q}^*}^{-1} \left(\vect{y} - \vect{m}_{\vect{\beta}^*(\vect{q}^*)}\right) \right\|^2_2.

This leads to a further reduction of the log-likelihood function where both :math:`\vect{\beta}`
and :math:`\sigma` are replaced by their expression in terms of :math:`\vect{q}`.

The default optimizer is :class:`~openturns.TNC` and can be changed thanks to the *setOptimizationAlgorithm* method.
User could also change the default optimization solver by setting the `GeneralLinearModelAlgorithm-DefaultOptimizationAlgorithm`
resource map key to one of the :class:`~openturns.NLopt` solver names.

It is also possible to proceed as follows:

- ask for the reduced log-likelihood function of the *GeneralLinearModelAlgorithm* thanks to the *getObjectiveFunction()* method
- optimize it with respect to the parameters :math:`\vect{\theta}` and  :math:`\vect{\sigma}` using any
  optimization algorithms (that can take into account some additional constraints if needed)
- set the optimal parameter value into the covariance model used in the *GeneralLinearModelAlgorithm*
- tell the algorithm not to optimize the parameter using *setOptimizeParameters*

The behaviour of the reduction is controlled by the following keys in :class:`~openturns.ResourceMap`:

- *ResourceMap.SetAsBool('GeneralLinearModelAlgorithm-UseAnalyticalAmplitudeEstimate', True)* to use the reduction associated to :math:`\sigma`. It has no effect if :math:`\outputDim > 1` or if :math:`\outputDim = 1` and :math:`\sigma` is not part of :math:`\vect{p}`
- *ResourceMap.SetAsBool('GeneralLinearModelAlgorithm-UnbiasedVariance', True)* allows one to use the *unbiased* estimate of :math:`\sigma` where :math:`\dfrac{1}{\sampleSize}` is replaced by :math:`\dfrac{1}{\sampleSize - d_p}` in the optimality condition for :math:`\sigma`, where :math:`d_p \in \Nset` is the dimension of the vector :math:`\vect{p}`.

With huge samples, the `hierarchical matrix <http://en.wikipedia.org/wiki/Hierarchical_matrix>`_  implementation could be used if OpenTURNS had been compiled with `hmat-oss` support.

This implementation, which is based on a compressed representation of an approximated covariance matrix (and its Cholesky factor),
has a better complexity both in terms of memory requirements and floating point operations.
To use it, the `GeneralLinearModelAlgorithm-LinearAlgebra` resource map key should be set to `HMAT`. Default value of the key is `LAPACK`.

A known centered gaussian observation noise :math:`\epsilon_k` can be taken into account
with :func:`setNoise()`:

.. math::
    \widehat{\vect{y}}_k 
    = \vect{y}_k + \epsilon_k, \quad \epsilon_k \sim \mathcal{N}\left(0, \tau_k^2\right)

Examples
--------
Create the model :math:`\model: \Rset \mapsto \Rset` and the samples:

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x'], ['x+x * sin(x)'])
>>> inputSample = ot.Sample([[1.0], [3.0], [5.0], [6.0], [7.0], [8.0]])
>>> outputSample = f(inputSample)

Create the algorithm:

>>> f1 = ot.SymbolicFunction(['x'], ['sin(x)'])
>>> f2 = ot.SymbolicFunction(['x'], ['x'])
>>> f3 = ot.SymbolicFunction(['x'], ['cos(x)'])
>>> basis = ot.Basis([f1,f2, f3])
>>> covarianceModel = ot.SquaredExponential([1.0])
>>> covarianceModel.setActiveParameter([])
>>> algo = ot.GeneralLinearModelAlgorithm(inputSample, outputSample, covarianceModel, basis)
>>> algo.run()

Get the resulting meta model:

>>> result = algo.getResult()
>>> metamodel = result.getMetaModel()
)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getResult
"Get the results of the metamodel computation.

Returns
-------
result : :class:`~openturns.GeneralLinearModelResult`
    Structure containing all the results obtained after computation
    and created by the method :py:meth:`run`.
"

//-----------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getInputSample
R"RAW(Accessor to the input sample.

Returns
-------
inputSample : :class:`~openturns.Sample`
    The input sample :math:`(\vect{x}_k)_{1 \leq k \leq \sampleSize}`.)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getOutputSample
R"RAW(Accessor to the output sample.

Returns
-------
outputSample : :class:`~openturns.Sample`
    The output sample :math:`(\vect{y}_k)_{1 \leq k \leq \sampleSize}` .)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getObjectiveFunction()
R"RAW(Accessor to the log-likelihood function that writes as argument of the covariance's model parameters.

Returns
-------
logLikelihood : :class:`~openturns.Function`
    The log-likelihood function degined in :eq:`logLikelihood` as a function of :math:`(\vect{\theta}, \vect{\sigma})`.

Notes
-----
The log-likelihood function may be useful for some postprocessing: maximization using external optimizers for example.


Examples
--------
Create the model :math:`\model: \Rset \mapsto \Rset` and the samples:

>>> import openturns as ot
>>> f = ot.SymbolicFunction(['x0'], ['x0 * sin(x0)'])
>>> inputSample = ot.Sample([[1.0], [3.0], [5.0], [6.0], [7.0], [8.0]])
>>> outputSample = f(inputSample)

Create the algorithm:

>>> basis = ot.ConstantBasisFactory().build()
>>> covarianceModel = ot.SquaredExponential(1)
>>> algo = ot.GeneralLinearModelAlgorithm(inputSample, outputSample, covarianceModel, basis)
>>> algo.run()

Get the log-likelihood function:

>>> likelihoodFunction = algo.getObjectiveFunction()
)RAW"


// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::run
"Compute the response surface.

Notes
-----
It computes the response surface and creates a
:class:`~openturns.GeneralLinearModelResult` structure containing all the results."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getOptimizationAlgorithm
"Accessor to solver used to optimize the covariance model parameters.

Returns
-------
algorithm : :class:`~openturns.OptimizationAlgorithm`
    Solver used to optimize the covariance model parameters.
    Default optimizer is :class:`~openturns.TNC`"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::setOptimizationAlgorithm
"Accessor to the solver used to optimize the covariance model parameters.

Parameters
----------
algorithm : :class:`~openturns.OptimizationAlgorithm`
    Solver used to optimize the covariance model parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::setOptimizeParameters
"Accessor to the covariance model parameters optimization flag.

Parameters
----------
optimizeParameters : bool
    Whether to optimize the covariance model parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getOptimizeParameters
"Accessor to the covariance model parameters optimization flag.

Returns
-------
optimizeParameters : bool
    Whether to optimize the covariance model parameters."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::setOptimizationBounds
"Optimization bounds accessor.

Parameters
----------
bounds : :class:`~openturns.Interval`
    Bounds for covariance model parameter optimization.

Notes
-----
Parameters involved by this method are:

 - Scale parameters,
 - Amplitude parameters if output dimension is greater than one or
   analytical sigma disabled,
 - Additional parameters.

Lower & upper bounds are defined in resource map.
Default lower upper bounds value for all parameters is
:math:`10^{-2}` and defined thanks to the
`GeneralLinearModelAlgorithm-DefaultOptimizationLowerBound`
resource map key.

For scale parameters, default upper bounds are set as :math:`2`
times the difference between the max and min values of `X` for
each coordinate, `X` being the (transformed) input sample.
The value :math:`2` is defined in resource map
(`GeneralLinearModelAlgorithm-DefaultOptimizationScaleFactor`).

Finally for other parameters (amplitude,...), default upper bound is set
to :math:`100` (corresponding resource map key is
`GeneralLinearModelAlgorithm-DefaultOptimizationUpperBound`)
"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getOptimizationBounds
"Optimization bounds accessor.

Returns
-------
bounds : :class:`~openturns.Interval`
    Bounds for covariance model parameter optimization."

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::setNoise
R"RAW(Observation noise variance accessor.

Parameters
----------
noise : sequence of positive float
    The noise variance :math:`\tau_k^2` of each output value.)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::GeneralLinearModelAlgorithm::getNoise
R"RAW(Observation noise variance accessor.

Parameters
----------
noise : sequence of positive float
    The noise variance :math:`\tau_k^2` of each output value.)RAW"
