module Msf
  ###
  #
  # This module exposes methods for querying a remote LDAP service
  #
  ###
  module Exploit::Remote::LDAP::ActiveDirectory

    module SecurityDescriptorMatcher
      CERTIFICATE_ENROLLMENT_EXTENDED_RIGHT = '0e10c968-78fb-11d2-90d4-00c04f79dc55'.freeze
      CERTIFICATE_AUTOENROLLMENT_EXTENDED_RIGHT = 'a05b8cc2-17bc-4802-a710-e7c15ab866a2'.freeze

      # This is a Base matcher class that can be used while analyzing security descriptors. It abstracts away the
      # checking of the permissions but relies on an external source to identify when particular ACEs will be in effect
      # based on a target principal SID.
      class Base
        # Check the ACE and determine if it should be ignored while processing. This allows processing to skip querying
        # the LDAP server when it's known that the ACE is irrelevant.
        #
        # @param [Rex::Proto::MsDtyp::MsDtypAce] ace The ace to check.
        def ignore_ace?(ace)
          false
        end

        # Apply the specified ACE to the internal state of the matcher because it will be applied in the hypothetical
        # access operation that is being analyzed.
        #
        # @param [Rex::Proto::MsDtyp::MsDtypAce] ace The ace to apply.
        # @rtype Nil
        def apply_ace!(ace)
          nil
        end

        # The matcher is satisfied when it has all the information it needs from previous calls to #apply_ace! to make
        # a determination with #matches?.
        def satisfied?
          false
        end

        # The matcher matches when it is satisfied and confident that the desired affect will be applied in the
        # hypothetical access operation.
        def matches?
          false
        end
      end

      # A general purpose Security Descriptor matcher for permissions in a single ACE. You typically want to use this to
      # check for a single permission.
      class Allow < Base
        attr_reader :permissions

        # @param [Symbol, Array<Symbol>] permissions The abbreviated permission names e.g. :WP.
        # @param [String] object_id An optional object GUID to use when matching the permission.
        def initialize(permissions, object_id: nil)
          @permissions = Array.wrap(permissions)
          @object_id = object_id
          @result = nil
        end

        def ignore_ace?(ace)
          # ignore anything that's not an allow or deny ACE because those are the only two that will alter the outcome
          return true unless Rex::Proto::MsDtyp::MsDtypAceType.allow?(ace.header.ace_type) || Rex::Proto::MsDtyp::MsDtypAceType.deny?(ace.header.ace_type)

          if Rex::Proto::MsDtyp::MsDtypAceType.has_object?(ace.header.ace_type) && ace.body.flags.ace_object_type_present == 1
            return true if ace.body.object_type != @object_id
          else
            return true if @object_id
          end

          ace_permissions = ace.body.access_mask.permissions
          !@permissions.all? { |perm| ace_permissions.include?(perm) }
        end

        def apply_ace!(ace)
          return if ignore_ace?(ace)

          @result = ace.header.ace_type

          nil
        end

        def satisfied?
          # A matcher is satisfied when there's nothing left for it to check.
          !@result.nil?
        end

        def matches?
          # This is named matches? instead of allow? so that other matchers can be made in the future
          # to match on the desired outcome including audit and alarm events. We are affirming that the
          # security descriptor will apply the desired affect which in this case is to allow access.
          satisfied? && Rex::Proto::MsDtyp::MsDtypAceType.allow?(@result)
        end

        # Build a matcher that will check for any of the specified permissions.
        #
        # @param [Array<Symbol>] The permissions to check for in their 2 letter abbreviated format, e.g. WP.
        # @param [String,Nil] An optional object ID that will be used for matching all the permissions.
        # @rtype MultipleAny
        def self.any(permissions, object_id: nil)
          permissions = Array.wrap(permissions)
          MultipleAll.new(permissions.map { |permission| new(permission, object_id: object_id) })
        end

        # Build a matcher that will check for all of the specified permissions.
        #
        # @param [Array<Symbol>] The permissions to check for in their 2 letter abbreviated format, e.g. WP.
        # @param [String,Nil] An optional object ID that will be used for matching all the permissions.
        # @rtype MultipleAll
        def self.all(permissions, object_id: nil)
          permissions = Array.wrap(permissions)
          MultipleAll.new(permissions.map { |permission| new(permission, object_id: object_id) })
        end

        def self.full_control
          # Full Control is special and shouldn't be split across multiple ACEs, so to check that we use #new instead of
          # MultipleAll to ensure it's in 1 ACE.
          new(%i[ CC DC LC SW RP WP DT LO CR SD RC WD WO ])
        end

        def self.certificate_autoenrollment
          MultipleAny.new([
            Allow.new(:CR, object_id: CERTIFICATE_AUTOENROLLMENT_EXTENDED_RIGHT),
            full_control
          ])
        end

        # Build a matcher that will check for a certificate's enrollment permission.
        def self.certificate_enrollment
          MultipleAny.new([
            Allow.new(:CR, object_id: CERTIFICATE_ENROLLMENT_EXTENDED_RIGHT),
            full_control
          ])
        end
      end

      # A compound matcher that will match when any of the sub-matchers match.
      class MultipleAny < Base
        attr_reader :matchers

        def initialize(matchers)
          @matchers = Array.wrap(matchers)
        end

        def ignore_ace?(ace)
          @matchers.all? { |matcher| matcher.ignore_ace?(ace) }
        end

        def apply_ace!(ace)
          @matchers.each do |matcher|
            next if matcher.ignore_ace?(ace)

            matcher.apply_ace!(ace)
          end

          nil
        end

        def satisfied?
          @matchers.any? { |matcher| matcher.satisfied? }
        end

        def matches?
          @matchers.any? { |matcher| matcher.matches? }
        end
      end

      # A compound matcher that will match when all of the sub-matchers match.
      class MultipleAll < MultipleAny
        def satisfied?
          @matchers.all? { |matcher| matcher.satisfied? }
        end

        def matches?
          @matchers.all? { |matcher| matcher.matches? }
        end
      end
    end
  end
end
