// license:BSD-3-Clause
// copyright-holders: David Haywood

/*

Space Bugger
Game-A-Tron, 1981

TS 20060821:
- there's really huge difference between both sets (code and data). weird
- wrong colors, it looks like there's direct color mapping - bits from attribute RAM connected to r/g/b outputs without
    use of color PROM or color table
- interrupt is generated by 8156 - very old combination chip containing 256*8 RAM (used as stack area),
  a timer and 8+8+6 ports. After game start all 8156 ports are set as inputs, timer is working in mode 3 - automatic reload.
  TC register is set to 0x7f. Timer input frequency is unknown. Output should be close to 1440 Hz.


PCB Layout
----------

CPU
2.81
GAT
  |------------------------------------------------------|
  |           74LS393                                    |
  |                         74LS00    TIP41A          CN1|
  |                                                      |
|-|           74LS32        74LS00    74LS161            |
|                                                        |
|                                     74LS10          CN2|
|    74LS139  74LS240                                    |
|                     |-----------|                      |
|    74LS240  74LS244 |  TMP8085  |   74LS04             |
|                     |-----------|6MHz                  |
|    74LS244  74LS373  74LS244  DSW(7)    74LS04         |
|                     |-----------|                      |
|    uPB8216  uPB8216 |  TMP8156  |                      |
|                     |-----------|                      |
|    2114  2114   74LS244  U32                           |
|                                                        |
|    2114  2114   74LS32   U33   U20                     |
|-|                                                      |
  |  2114  2114   74LS138  U34   U21                     |
  |                                                      |
  |  2114  2114   74LS138  U35   U22                     |
  |------------------------------------------------------|
Notes:
      All IC's shown.
      uPB8216 - NEC uPB8216 4 Bit Bi-directional Bus Driver
      2114    - Hitachi HM572114P-4 1k x4 SRAM
      TMP8156 - RIOT, 256 bytes Static RAM with I/O Ports and Timer
      TMP8085 - 8085 CPU, clock 3MHz [6/2]
      CN1/2   - 10 pin connectors
      U20-U22 - Hitachi HM462716G 2k x8 EPROMs
      U32-U35 /

      Note, this PCB plugs into a motherboard containing connectors. There is
      also a sound and graphics board that plugs into the motherboard as well.
      However those PCBs are not available and are therefore not documented.


 Set 2:
--------

Space Bugger
Game-A-Tron 1981


                       U20 U21 U22
                   U32 U33 U34 U35
             6MHz

          8085   8156
                       2114 2114 2114 2114
                       2114 2114 2114 2114

Main

-------


            GFX

            2114 2114 2114 2114

12.440MHz

Graphics PCB

-------



                         76489  76489


Sound PCB

-------

*/

#include "emu.h"

#include "cpu/i8085/i8085.h"
#include "machine/i8155.h"
#include "sound/sn76496.h"

#include "emupal.h"
#include "screen.h"
#include "speaker.h"
#include "tilemap.h"


namespace {

class sbugger_state : public driver_device
{
public:
	sbugger_state(const machine_config &mconfig, device_type type, const char *tag) :
		driver_device(mconfig, type, tag),
		m_maincpu(*this, "maincpu"),
		m_gfxdecode(*this, "gfxdecode"),
		m_videoram_attr(*this, "videoram_attr"),
		m_videoram(*this, "videoram")
	{ }

	void sbugger(machine_config &config);

protected:
	virtual void video_start() override ATTR_COLD;

private:
	required_device<cpu_device> m_maincpu;
	required_device<gfxdecode_device> m_gfxdecode;

	required_shared_ptr<uint8_t> m_videoram_attr;
	required_shared_ptr<uint8_t> m_videoram;

	tilemap_t *m_tilemap = nullptr;

	void videoram_w(offs_t offset, uint8_t data);
	void videoram_attr_w(offs_t offset, uint8_t data);

	TILE_GET_INFO_MEMBER(get_tile_info);

	void palette(palette_device &palette) const;

	uint32_t screen_update(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect);
	void io_map(address_map &map) ATTR_COLD;
	void prg_map(address_map &map) ATTR_COLD;
};


TILE_GET_INFO_MEMBER(sbugger_state::get_tile_info)
{
	int const tileno = m_videoram[tile_index];
	int const color = m_videoram_attr[tile_index];

	tileinfo.set(0,tileno,color,0);
}

void sbugger_state::videoram_w(offs_t offset, uint8_t data)
{
	m_videoram[offset] = data;
	m_tilemap->mark_tile_dirty(offset);
}

void sbugger_state::videoram_attr_w(offs_t offset, uint8_t data)
{
	m_videoram_attr[offset] = data;
	m_tilemap->mark_tile_dirty(offset);
}

void sbugger_state::video_start()
{
	m_tilemap = &machine().tilemap().create(*m_gfxdecode, tilemap_get_info_delegate(*this, FUNC(sbugger_state::get_tile_info)), TILEMAP_SCAN_ROWS, 8, 16, 64, 16);
}

uint32_t sbugger_state::screen_update(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect)
{
	m_tilemap->draw(screen, bitmap, cliprect, 0, 0);
	return 0;
}

// not right but so we can see things OK
void sbugger_state::palette(palette_device &palette) const
{
	// just some random colours for now
	for (int i = 0; i < 256; i++)
	{
		int const r = i ? (machine().rand() | 0x80) : 0;
		int const g = i ? (machine().rand() | 0x80) : 0;
		int const b = i ? (machine().rand() | 0x80) : 0;

		palette.set_pen_color(i * 2 + 1, rgb_t(r, g, b));
		palette.set_pen_color(i * 2, rgb_t(0, 0, 0));
	}
}


void sbugger_state::prg_map(address_map &map)
{
	map(0x0000, 0x37ff).rom();
	map(0xc800, 0xcbff).ram().w(FUNC(sbugger_state::videoram_attr_w)).share(m_videoram_attr);
	map(0xcc00, 0xcfff).ram().w(FUNC(sbugger_state::videoram_w)).share(m_videoram);
	map(0xe000, 0xe0ff).rw("i8156", FUNC(i8155_device::memory_r), FUNC(i8155_device::memory_w)); // sp is set to e0ff
	map(0xf400, 0xffff).ram();
}

void sbugger_state::io_map(address_map &map)
{
	map(0xe0, 0xe7).rw("i8156", FUNC(i8155_device::io_r), FUNC(i8155_device::io_w));
	map(0xe8, 0xe8).w("sn76489.1", FUNC(sn76489_device::write));
	map(0xe9, 0xe9).w("sn76489.2", FUNC(sn76489_device::write));
}


static const gfx_layout char16layout =
{
	8,16,
	RGN_FRAC(1,1),
	1,
	{ 0 },
	{ 0, 1, 2, 3, 4, 5, 6, 7 },
	{ 0*8, 1*8, 2*8, 3*8, 4*8, 5*8, 6*8, 7*8,
		8*8, 9*8, 10*8, 11*8, 12*8, 13*8, 14*8, 15*8
	},
	16*8
};

static GFXDECODE_START( gfx_sbugger )
	GFXDECODE_ENTRY( "chars", 0, char16layout,   0, 256  )
GFXDECODE_END


static INPUT_PORTS_START( sbugger )
	PORT_START("INPUTS")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_UP ) PORT_2WAY
	PORT_DIPNAME( 0x02, 0x02, "PA 1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_BUTTON1   )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN ) PORT_2WAY
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON2   )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_START1 )
	PORT_DIPNAME( 0x80, 0x80, "PA 7" )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START("DSW1")
	PORT_DIPNAME( 0x01, 0x01, "PB 0" )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, "PB 1" )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, "PB 2" )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "PB 3" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, "PB 4" )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, "PB 5" )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, "PB 6" )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_COIN1 )

	PORT_START("DSW2")
	PORT_DIPNAME( 0x03, 0x01, DEF_STR( Difficulty ) ) //enemy speed
	PORT_DIPSETTING(    0x00, DEF_STR( Easy ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Normal ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Hard ) )
	PORT_DIPSETTING(    0x03, DEF_STR( Hardest ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Allow_Continue ) )
	PORT_DIPSETTING(    0x00, DEF_STR( No ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Yes ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) ) //bonus stage related (code at $f94 & $df8)
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xc0, IP_ACTIVE_HIGH, IPT_UNUSED )
INPUT_PORTS_END


void sbugger_state::sbugger(machine_config &config)
{
	I8085A(config, m_maincpu, 6'000'000);        // 3.00 MHz???
	m_maincpu->set_addrmap(AS_PROGRAM, &sbugger_state::prg_map);
	m_maincpu->set_addrmap(AS_IO, &sbugger_state::io_map);

	i8156_device &i8156(I8156(config, "i8156", 200'000));     // freq is an approximation
	i8156.in_pa_callback().set_ioport("INPUTS");
	i8156.in_pb_callback().set_ioport("DSW1");
	i8156.in_pc_callback().set_ioport("DSW2");
	i8156.out_to_callback().set_inputline(m_maincpu, I8085_RST75_LINE);

	GFXDECODE(config, m_gfxdecode, "palette", gfx_sbugger);

	screen_device &screen(SCREEN(config, "screen", SCREEN_TYPE_RASTER));
	screen.set_refresh_hz(60);
	screen.set_vblank_time(ATTOSECONDS_IN_USEC(0));
	screen.set_size(64*8, 32*8);
	screen.set_visarea(0*8, 64*8-1, 0*8, 32*8-1);
	screen.set_screen_update(FUNC(sbugger_state::screen_update));
	screen.set_palette("palette");

	PALETTE(config, "palette", FUNC(sbugger_state::palette), 512);

	// sound hardware
	SPEAKER(config, "mono").front_center();

	SN76489(config, "sn76489.1", 3'000'000).add_route(ALL_OUTPUTS, "mono", 1.0);

	SN76489(config, "sn76489.2", 3'000'000).add_route(ALL_OUTPUTS, "mono", 1.0);
}


ROM_START( sbugger )
	ROM_REGION( 0x10000, "maincpu", 0 ) // 8085 code
	ROM_LOAD( "spbugger.u35", 0x0000, 0x0800, CRC(7c2000a1) SHA1(01a60745ea8e9a70de37d1a785fad1d17eafc812) ) // seems to map at 0
	ROM_LOAD( "spbugger.u22", 0x0800, 0x0800, BAD_DUMP CRC(66e00c53) SHA1(49ca567a98978308306cdb8455c61c022668693b) ) // FIXED BITS (xxxx1111)  it jumps here .... bad ROM?
	ROM_LOAD( "spbugger.u34", 0x1000, 0x0800, CRC(db357dde) SHA1(363392b971f48e9d99f4167aa17f0c885b0865ee) ) // seems to map at 1000
	ROM_LOAD( "spbugger.u21", 0x1800, 0x0800, CRC(618a5b2a) SHA1(aa7a40b1944f09c396f675d7dd3a8c3c35bf01f1) ) // seems to map at 1800
	ROM_LOAD( "spbugger.u20", 0x2000, 0x0800, CRC(8957563c) SHA1(b33a75fcf375d2a1a766105f87dd8e4d42db3d76) ) // seems to map at 2000
	ROM_LOAD( "spbugger.u33", 0x2800, 0x0800, CRC(f6cb1399) SHA1(53cb67e29a238c5ac20c6be9423d850e004212c1) ) // seems to map at 2800
	ROM_LOAD( "spbugger.u32", 0x3000, 0x0800, CRC(f49af2b3) SHA1(1519ee4786b78546767827d3a9508e7ddb646765) ) // seems to map at 3000

	ROM_REGION( 0x1000, "chars", 0 )
	ROM_LOAD( "spbugger.gfx", 0x0000, 0x1000, CRC(d3f345b5) SHA1(a5082ffc3043352e9b731af95770bdd62fb928bf) )
ROM_END

ROM_START( sbuggera )
	ROM_REGION( 0x10000, "maincpu", 0 ) // 8085 code
	ROM_LOAD( "bug_g10.u35", 0x0000, 0x0800, CRC(60a3044d) SHA1(5b2be551a84e4a7a35637208a19b3477629f20d9) )
	ROM_LOAD( "bug_c10.u22", 0x0800, 0x0800, CRC(34a829f7) SHA1(135ec2739879c2e47f3c6d4a5196c865b5940a84) )
	ROM_LOAD( "bug_f10.u34", 0x1000, 0x0800, CRC(e2f7a51c) SHA1(ee221f6697021d14838fd6c4aff41678ce62e4ba) )
	ROM_LOAD( "bug_b10.u21", 0x1800, 0x0800, CRC(3c9a467a) SHA1(be64fddc294be38ff4f997e6666aaea15ed7d0e9) )
	ROM_LOAD( "bug_a10.u20", 0x2000, 0x0800, CRC(f6f58aba) SHA1(95bf03b8876ff36f52d93f5f5f9b38021c8b7979) )
	ROM_LOAD( "bug_e10.u33", 0x2800, 0x0800, CRC(fefd9c5e) SHA1(1b0bbf462231c32014d45ec21b105a669665d90d) )
	ROM_LOAD( "bug_d10.u32", 0x3000, 0x0800, BAD_DUMP CRC(c807742f) SHA1(cb5c44ffd6dd184c6a0722003ff6674caf865bee) ) // missing in this set

	ROM_REGION( 0x4000, "chars", 0 )
	ROM_LOAD( "spbugger.gfx", 0x0000, 0x1000, CRC(d3f345b5) SHA1(a5082ffc3043352e9b731af95770bdd62fb928bf) )
ROM_END

} // anonymous namespace


GAME( 1981, sbugger,  0,        sbugger,  sbugger, sbugger_state, empty_init, ROT270, "Game-A-Tron", "Space Bugger (set 1)", MACHINE_NOT_WORKING | MACHINE_WRONG_COLORS | MACHINE_SUPPORTS_SAVE )
GAME( 1981, sbuggera, sbugger,  sbugger,  sbugger, sbugger_state, empty_init, ROT270, "Game-A-Tron", "Space Bugger (set 2)", MACHINE_WRONG_COLORS | MACHINE_SUPPORTS_SAVE )
